//=============================================================================
// Copyright  Micromega Corporation 2005-2007. All rights reserved.
//
// @file	GCdistance.c
// @target	Atmel WinAVR C compiler
//
// see: Application Note 39 - Calculating Great Circle Distances
//
// This program demonstrates the calculation of great circle distances.
// It reads the current latitude and longitude position from a GPS receiver,
// then calculates the great circle distance to a number of stored locations.
// The program can also be run without a GPS connection by selecting one of
// the pre-defined locations as the 'from' location.
//
// @author	Cam Thompson, Micromega Corporation
// @version
//	October 15, 2007
//	- original version
//=============================================================================

#include <avr/io.h>
#include <stdio.h>
#include "uart.h"
#include "fpu.h"

FILE uart_stream = FDEV_SETUP_STREAM(uart_putchar, NULL, _FDEV_SETUP_WRITE);

//------------------ local prototypes -----------------------------------------

//------------------ uM-FPU register definitions ------------------------------
#define lat1			10			// uM-FPU register 10
#define long1			11			// uM-FPU register 11
#define lat2			12			// uM-FPU register 12
#define long2  			13			// uM-FPU register 13

#define dist_nm			20			// distance - nautical miles
#define dist_km			21			// distance - kilometers
#define dist_miles		22			// distance - miles

  //-------------------- uM-FPU Function Definitions --------------------------
#define getID			0			// uM-FPU user function 0
#define getDistance		1			// uM-FPU user function 1
#define getLocation		2			// uM-FPU user function 2
#define getLatLong		3			// uM-FPU user function 3
#define radiansToDM		4			// uM-FPU user function 4
#define readNMEA		5			// uM-FPU user function 5
#define parseGPRMC		6			// uM-FPU user function 6
#define NMEA_Degrees	7			// uM-FPU user function 7

//------------------ main -----------------------------------------------------

int main(void) {
	char from, index;

	// initialize the UART and set as stdout
	uart_init();
	stdout = &uart_stream;

	// print program name
	printf("\r\n\r\nGreat Circle Distance\r\n");

	// Reset the uM-FPU and print version.
	if (fpu_reset() == SYNC_CHAR) {
		print_version();
		print_CRLF();
	}
	else {
		printf("uM-FPU not detected");
		return(0);
	}

	// check that the user-defined functions are loaded
	fpu_write3(CLR0, FCALL, getID);
	fpu_write2(SELECTA, 0);

	fpu_wait();
	fpu_write(LREADBYTE);
	if (fpu_read() != 39) {
		printf("GCdistance.fpu functions not loaded.");
		return(0);
	}

	// if from = 99, read current location from GPS
	// if from <> 99, use a stored location (0 to 7)
	// ---------------------------------------------
	from = 0;

	printf("\r\nFrom:\r\n  ");
	if (from == 99) {
		printf("Reading GPS ...");
		fpu_write2(FCALL, readNMEA);
		fpu_wait();

		fpu_write(STRSET);
		fpu_writeString("Location");
	}
	else {
		fpu_write4(LONGBYTE, from, FCALL, getLocation);
		fpu_write3(COPY, lat2, lat1);
		fpu_write3(COPY, long2, long1);
	}

	// display latitude and longitude
	// ------------------------------
	fpu_write(STRINS);
	fpu_writeString(" (");
	fpu_write2(SELECTA, lat1);
	fpu_write3(FCALL, radiansToDM, STRINS);
    fpu_writeString(", ");
	fpu_write2(SELECTA, long1);
	fpu_write3(FCALL, radiansToDM, STRINS);
	fpu_writeString(")");
	print_fpuString(READSTR);

	// display distance to all other stored locations
	// ----------------------------------------------
	printf("\r\n\r\nTo:");

	for(index = 0; index < 8; index++) {
		if (from != index) {

			printf("\r\n  ");
			fpu_write2(LONGBYTE, index);
			fpu_write3(FCALL, getLocation, STRINS);
			fpu_writeString(" (");
			fpu_write2(SELECTA, lat2);
			fpu_write3(FCALL, radiansToDM, STRINS);
			fpu_writeString(", ");
			fpu_write2(SELECTA, long2);
			fpu_write3(FCALL, radiansToDM, STRINS);
			fpu_writeString(")");
			print_fpuString(READSTR);
			print_CRLF();

 			printf("   ");
			fpu_write2(SELECTA, dist_nm);
			fpu_write3(FCALL, getDistance, FSET0);
			print_float(92);
			printf(" nautical miles\r\n");

			printf("   ");
			fpu_write4(SELECTA, dist_km, FSET, dist_nm);
			fpu_write(FWRITE0);
			fpu_writeFloat(1.852);
			fpu_write(FMUL0);
			fpu_write4(SELECTA, dist_miles, FSET, dist_km);
			fpu_write2(FCNV, 13);
			print_float(92);
			printf(" miles\r\n");
		  	  
			printf("   ");
			fpu_write2(SELECTA, dist_km);
			print_float(92);
			printf(" kilometers\r\n");
        }
    }

	printf("\r\nDone.\r\n");
	return 0;
}
